"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.mock = void 0;
var lodash_1 = __importDefault(require("lodash"));
function resolveAllOf(schema) {
    if (schema.allOf && schema.allOf[0]) {
        schema = lodash_1.default.reduce(schema.allOf, function (combined, subschema) { return lodash_1.default.merge({}, combined, resolveAllOf(subschema)); }, schema);
    }
    return schema;
}
function mock(schema) {
    // allOf, merge all subschemas
    schema = resolveAllOf(schema);
    // use specified example
    if (schema.example !== undefined) {
        return schema.example;
    }
    // use default
    if (schema.default !== undefined) {
        return schema.default;
    }
    // oneOf, use first
    if (schema.oneOf && schema.oneOf[0]) {
        return mock(schema.oneOf[0]);
    }
    // anyOf, use first
    if (schema.anyOf && schema.anyOf[0]) {
        return mock(schema.anyOf[0]);
    }
    // get type, use first if array
    var type = lodash_1.default.isArray(schema) ? lodash_1.default.first(schema.type) : schema.type;
    if (type === 'object') {
        var obj = schema;
        var properties = obj.properties;
        if (!properties) {
            return {};
        }
        return lodash_1.default.mapValues(properties, mock);
    }
    if (type === 'array') {
        var array = schema;
        var items = array.items;
        if (!items) {
            return [];
        }
        var examples = [];
        var example = ((items.oneOf && items.oneOf[0]) || items);
        if (items.anyOf) {
            // include one of each example for anyOf and allOf
            for (var _i = 0, _a = items.anyOf; _i < _a.length; _i++) {
                var option = _a[_i];
                example = option;
                examples.push(mock(example));
            }
        }
        // if minItems is set make sure we have at least that many items
        var minItems = array.minItems || 1;
        while (examples.length < minItems) {
            examples.push(mock(example));
        }
        // limit to max items if applicable
        return array.maxItems ? lodash_1.default.take(examples, array.maxItems) : examples;
    }
    if (lodash_1.default.isArray(schema.enum)) {
        return schema.enum[0];
    }
    if (type === 'string') {
        var format = schema.format;
        var formatExamples = {
            email: 'user@example.com',
            hostname: 'example.com',
            ipv4: '8.8.8.8',
            ipv6: '2001:4860:4860::8888',
            uri: 'https://example.com/path',
            'uri-reference': '/path#anchor',
            'uri-template': '/path/{param}',
            'json-pointer': '/foo/bar',
            'date-time': new Date('1970-01-01').toJSON(),
            uuid: '3fa85f64-5717-4562-b3fc-2c963f66afa6',
            _default: 'string',
        };
        var val = formatExamples[format] || formatExamples._default;
        var minln = !lodash_1.default.isNil(schema.minLength) ? schema.minLength : 0;
        var maxln = !lodash_1.default.isNil(schema.maxLength) ? schema.maxLength : val.length;
        if (val === formatExamples._default && val.length < minln) {
            return lodash_1.default.padEnd(val, minln, val);
        }
        return val.substr(0, lodash_1.default.clamp(val.length, minln, maxln));
    }
    if (type === 'number') {
        var min = schema.minimum ? schema.minimum : -Number.MAX_VALUE;
        var max = schema.maximum ? schema.maximum : Number.MAX_VALUE;
        if (schema.multipleOf) {
            min = Math.ceil(min / schema.multipleOf) * schema.multipleOf;
            max = Math.floor(max / schema.multipleOf) * schema.multipleOf;
        }
        return lodash_1.default.clamp(0, min, max);
    }
    if (type === 'integer') {
        var schemaMin = schema.minimum && schema.exclusiveMinimum ? schema.minimum + 1 : schema.minimum;
        var schemaMax = schema.maximum && schema.exclusiveMaximum ? schema.maximum - 1 : schema.maximum;
        var min = !lodash_1.default.isNil(schemaMin) ? schemaMin : Number.MIN_SAFE_INTEGER;
        var max = !lodash_1.default.isNil(schemaMax) ? schemaMax : Number.MAX_SAFE_INTEGER;
        if (schema.multipleOf) {
            min = Math.ceil(min / schema.multipleOf) * schema.multipleOf;
            max = Math.floor(max / schema.multipleOf) * schema.multipleOf;
        }
        return lodash_1.default.clamp(0, min, max);
    }
    if (type === 'null') {
        return null;
    }
    if (type === 'boolean') {
        return true;
    }
    // unknown type
    return {};
}
exports.mock = mock;
//# sourceMappingURL=mock.js.map